<?php
// admin/clear_exam_data.php - Clears all exam data for a specific session, allowing retake.

require_once '../db.php';
require_admin();

if (!isset($_GET['session_id']) || !is_numeric($_GET['session_id'])) {
    redirect('admin/test_takers.php', 'error', 'Invalid session ID.');
}

$session_id = (int)$_GET['session_id'];

// 1. Fetch data needed for file deletion before deleting the session record
$sql_fetch_data = "SELECT public_session_id, proctoring_video_path FROM student_exam_sessions WHERE session_id = ?";
$result_data = execute_query($mysqli, $sql_fetch_data, 'i', [$session_id]);
$session_data = $result_data ? $result_data->fetch_assoc() : null;

if (!$session_data) {
    redirect('admin/test_takers.php', 'error', 'Exam session not found.');
}

$public_session_id = $session_data['public_session_id'];
$video_path = $session_data['proctoring_video_path'];

// 2. Start Transaction for database integrity
$mysqli->begin_transaction();
$success = true;

try {
    // 3. Delete student answers
    $sql_delete_answers = "DELETE FROM student_answers WHERE session_id = ?";
    if (!execute_non_query($mysqli, $sql_delete_answers, 'i', [$session_id])) {
        throw new Exception("Failed to delete student answers.");
    }
    
    // 4. Delete video chunk records
    $sql_delete_chunks = "DELETE FROM exam_video_chunks WHERE session_id = ?";
    if (!execute_non_query($mysqli, $sql_delete_chunks, 'i', [$session_id])) {
        throw new Exception("Failed to delete video chunk records.");
    }

    // 5. Delete the exam session record
    $sql_delete_session = "DELETE FROM student_exam_sessions WHERE session_id = ?";
    if (!execute_non_query($mysqli, $sql_delete_session, 'i', [$session_id])) {
        throw new Exception("Failed to delete exam session.");
    }

    // 5. Commit transaction
    $mysqli->commit();

    // 6. Attempt to delete associated files (outside of transaction)
    $files_deleted = 0;
    
    // Delete final video file
    if (!empty($video_path)) {
        // The path is relative to the project root, so we prepend PROJECT_ROOT_DIR
        $full_video_path = PROJECT_ROOT_DIR . $video_path;
        if (file_exists($full_video_path)) {
            $command = "rm -f " . escapeshellarg($full_video_path);
            exec($command, $output, $return_var);
            if ($return_var === 0) {
                $files_deleted++;
            } else {
                error_log("Failed to delete final video file: {$full_video_path}. Command output: " . implode("\n", $output));
            }
        }
    }

    // Delete video processing log file (assuming log name is based on session_id)
    $log_path = PROJECT_ROOT_DIR . 'videos/processing_log_' . $session_id . '.txt';
    if (file_exists($log_path) && unlink($log_path)) {
        $files_deleted++;
    }
    
    // Delete video chunks directory
    // The chunk directory is named after the public_session_id (which is the username) and is located inside VIDEO_CHUNKS_DIR
    // Note: public_session_id is the username in this application.
    $chunk_dir = VIDEO_CHUNKS_DIR . '/' . $public_session_id;
    if (is_dir($chunk_dir)) {
        // Use shell command for recursive directory deletion
        // We use 'rm -rf' to ensure all contents and the directory itself are removed
        $command = "rm -rf " . escapeshellarg($chunk_dir);
        exec($command, $output, $return_var);
        if ($return_var === 0) {
            $files_deleted++; // Count the directory deletion as one file operation
        } else {
            error_log("Failed to delete chunk directory: {$chunk_dir}. Command output: " . implode("\n", $output));
        }
    }

    redirect('admin/test_takers.php', 'success', "Exam data for session ID {$session_id} cleared successfully. {$files_deleted} associated files/directories removed.");

} catch (Exception $e) {
    $mysqli->rollback();
    error_log("Failed to clear exam data: " . $e->getMessage());
    redirect('admin/test_takers.php', 'error', 'Failed to clear exam data: ' . $e->getMessage());
}

$mysqli->close();
?>
