<?php
// manual_grading_list.php - List of completed exam sessions that require manual theory grading

require_once '../db.php';
require_admin();

$page_title = 'Manual Grading List';

// SQL to find all completed sessions that have at least one Theory question
// and where the score for that question is still 0 (meaning it hasn't been graded yet)
$sql_sessions = "
    SELECT 
        S.session_id, 
        U.full_name, 
        E.exam_title, 
        S.total_score,
        S.start_time,
        S.end_time,
        (SELECT COUNT(SA.question_id) 
         FROM student_answers SA 
         JOIN exam_questions EQ ON SA.question_id = EQ.question_id
         WHERE SA.session_id = S.session_id AND EQ.question_type = 'Theory' AND SA.score_earned = 0) AS ungraded_theory_count,
        (SELECT COUNT(SA.question_id) 
         FROM student_answers SA 
         JOIN exam_questions EQ ON SA.question_id = EQ.question_id
         WHERE SA.session_id = S.session_id AND EQ.question_type = 'Theory') AS total_theory_count
    FROM student_exam_sessions S
    JOIN users U ON S.student_id = U.user_id
    JOIN exams E ON S.exam_id = E.exam_id
    WHERE S.is_submitted = 1
    HAVING total_theory_count > 0
    ORDER BY S.end_time DESC
";

$result = execute_query($mysqli, $sql_sessions);
$sessions = $result ? $result->fetch_all(MYSQLI_ASSOC) : [];

include 'includes/header.php';
?>

<div class="admin-content">
    <h2><?php echo $page_title; ?></h2>
    <p>List of completed exam sessions that contain Theory questions and require manual grading.</p>

    <?php if (empty($sessions)): ?>
        <p class="info">No exam sessions currently require manual theory grading.</p>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>Session ID</th>
                    <th>Student Name</th>
                    <th>Exam Title</th>
                    <th>Submission Time</th>
                    <th>Theory Questions</th>
                    <th>Current Score</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($sessions as $session): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($session['session_id']); ?></td>
                        <td><?php echo htmlspecialchars($session['full_name']); ?></td>
                        <td><?php echo htmlspecialchars($session['exam_title']); ?></td>
                        <td><?php echo htmlspecialchars($session['end_time']); ?></td>
                        <td>
                            <?php 
                                $status_class = $session['ungraded_theory_count'] > 0 ? 'status-pending' : 'status-completed';
                                $status_text = $session['ungraded_theory_count'] > 0 ? 'Pending Grading' : 'Grading Complete';
                            ?>
                            <span class="<?php echo $status_class; ?>">
                                <?php echo htmlspecialchars($session['total_theory_count'] - $session['ungraded_theory_count']); ?> / <?php echo htmlspecialchars($session['total_theory_count']); ?> Graded (<?php echo $status_text; ?>)
                            </span>
                        </td>
                        <td><?php echo number_format($session['total_score'], 2); ?></td>
                        <td>
                            <a href="grade_theory.php?session_id=<?php echo $session['session_id']; ?>" class="btn btn-primary btn-small">Grade / Review</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<?php
include 'includes/footer.php';
?>
