<?php
// db.php - Database Connection and Configuration for Proctored Exam System

// =================================================================================================
// 1. START SESSION AND AUTOLOAD
// =================================================================================================
session_start();

// Include Composer Autoloader (for PHPMailer)
require __DIR__ . '/vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// =================================================================================================
// 2. BASE URL AND PATHS
// =================================================================================================

// Base URL for local XAMPP setup (adjust if project folder is different)
if (!defined('BASE_URL')) {
    define('BASE_URL', 'http://localhost/proctor/proctor/'); // ends with slash
}

// Project root directory
if (!defined('PROJECT_ROOT_DIR')) {
    define('PROJECT_ROOT_DIR', __DIR__ . '/');
}

// Video chunk directories
define('VIDEO_CHUNKS_DIR', PROJECT_ROOT_DIR . 'chunks/');
if (!is_dir(VIDEO_CHUNKS_DIR))
    mkdir(VIDEO_CHUNKS_DIR, 0777, true);

define('FINAL_VIDEO_DIR', PROJECT_ROOT_DIR . 'final_videos/');
if (!is_dir(FINAL_VIDEO_DIR))
    mkdir(FINAL_VIDEO_DIR, 0777, true);

// =================================================================================================
// 3. WORKER & UPLOAD HANDLER PATHS (ABSOLUTE URLS)
// =================================================================================================
define('WORKER_SCRIPT_PATH', BASE_URL . 'upload_worker.js');
define('CHUNK_UPLOAD_HANDLER', BASE_URL . 'upload_chunk.php');
define('FINALIZE_UPLOAD_HANDLER', BASE_URL . 'finalize_upload.php');
define('EXAM_SUBMIT_HANDLER', BASE_URL . 'student/submit_exam.php'); // adjust if necessary

// =================================================================================================
// 4. PROCTORING SETTINGS
// =================================================================================================
define('MAX_WORKERS', 5);           // concurrent worker threads
define('CHUNK_DURATION_MS', 1000);  // milliseconds per chunk

// =================================================================================================
// 5. DATABASE CONFIGURATION
// =================================================================================================
define('DB_SERVER', 'localhost');
define('DB_USERNAME', 'root'); // your DB username
define('DB_PASSWORD', '');     // your DB password
define('DB_NAME', 'proctored_exam_app'); // your DB name

$mysqli = new mysqli(DB_SERVER, DB_USERNAME, DB_PASSWORD, DB_NAME);
if ($mysqli->connect_error) {
    error_log("Database Connection Failed: " . $mysqli->connect_error);
}

// =================================================================================================
// 6. PROJECT INFO
// =================================================================================================
define('PROJECT_NAME', 'Proctored Exam System');
define('MAIN_JS_PATH', BASE_URL . 'assets/js/proctoring.js');
define('MAIN_CSS_PATH', BASE_URL . 'assets/css/style.css');

// =================================================================================================
// 7. PHPMailer CONFIGURATION
// =================================================================================================
define('SMTP_HOST', 'smtp.example.com');
define('SMTP_USERNAME', 'user@example.com');
define('SMTP_PASSWORD', 'your_password');
define('SMTP_PORT', 587);
define('SMTP_FROM_EMAIL', 'no-reply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost'));
define('SMTP_FROM_NAME', PROJECT_NAME . ' Admin');

// =================================================================================================
// 8. UTILITY FUNCTIONS
// =================================================================================================

function redirect($url, $type = null, $message = null)
{
    $final_url = (strpos($url, 'http') === 0) ? $url : BASE_URL . $url;
    if ($type && $message) {
        $_SESSION['flash_type'] = $type;
        $_SESSION['flash_message'] = $message;
    }
    if (headers_sent()) {
        echo "<script>window.location.href='" . $final_url . "';</script>";
        exit;
    }
    header("Location: " . $final_url);
    exit;
}

function is_logged_in()
{
    return isset($_SESSION['user_id']);
}
function is_admin()
{
    return is_logged_in() && $_SESSION['user_type'] === 'Admin';
}
function is_student()
{
    return is_logged_in() && $_SESSION['user_type'] === 'Student';
}
function require_admin()
{
    if (!is_admin())
        redirect('index.php');
}
function require_student()
{
    if (!is_student())
        redirect('index.php');
}
function get_current_user_id()
{
    return $_SESSION['user_id'] ?? null;
}
function get_current_user_grade()
{
    return $_SESSION['grade'] ?? null;
}
function get_current_user_staff_id()
{
    return $_SESSION['staff_id'] ?? null;
}
function get_current_username()
{
    return $_SESSION['username'] ?? null;
}

// DB helper functions
function execute_query($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error)
        return false;
    $stmt = $mysqli->prepare($sql);
    if ($stmt) {
        if (!empty($types) && !empty($params))
            $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            error_log($stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log($mysqli->error);
        return false;
    }
}

function execute_non_query($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error)
        return false;
    $stmt = $mysqli->prepare($sql);
    if ($stmt) {
        if (!empty($types) && !empty($params))
            $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $affected = $stmt->affected_rows;
            $stmt->close();
            return $affected;
        } else {
            error_log($stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log($mysqli->error);
        return false;
    }
}

function execute_insert($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error)
        return false;
    $stmt = $mysqli->prepare($sql);
    if ($stmt) {
        if (!empty($types) && !empty($params))
            $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $id = $mysqli->insert_id;
            $stmt->close();
            return $id;
        } else {
            error_log($stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log($mysqli->error);
        return false;
    }
}

// PHPMailer email sending
function send_email($to, $subject, $body)
{
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = SMTP_PORT;
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($to);
        $mail->isHTML(false);
        $mail->Subject = $subject;
        $mail->Body = $body;
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Email failed to $to: {$e->getMessage()}");
        return false;
    }
}
?>