<?php
// upload_chunk.php - Handles parallel upload of video micro-chunks

require_once '../db.php';

// Enable detailed error reporting (for debugging)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set JSON headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Allow Worker uploads

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Only POST requests are allowed.']);
    exit;
}

// Check required POST params and uploaded file
if (!isset($_POST['session_id'], $_POST['chunk_index']) || !isset($_FILES['video_chunk'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing required parameters (session_id, chunk_index, video_chunk).']);
    exit;
}

$session_id = (int) $_POST['session_id'];
$chunk_index = (int) $_POST['chunk_index'];
$file = $_FILES['video_chunk'];

// --- 1. Validate session exists ---
$sql_session_check = "SELECT student_id FROM student_exam_sessions WHERE session_id = ?";
$result = execute_query($mysqli, $sql_session_check, 'i', [$session_id]);

if (!$result || $result->num_rows === 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid exam session ID.']);
    exit;
}

$session_data = $result->fetch_assoc();
$student_id = $session_data['student_id'];

// --- 2. Get username for this student ---
$sql_user = "SELECT username FROM users WHERE user_id = ?";
$result_user = execute_query($mysqli, $sql_user, 'i', [$student_id]);

if (!$result_user || $result_user->num_rows === 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'User not found for session.']);
    exit;
}

$user_data = $result_user->fetch_assoc();
$username = $user_data['username'];

// --- 3. Handle file upload ---
if ($file['error'] !== UPLOAD_ERR_OK) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'File upload error: ' . $file['error']]);
    exit;
}

// Base chunks directory is ensured to exist in db.php

// Ensure user subdirectory exists
$user_folder = VIDEO_CHUNKS_DIR . '/' . $username . '/';
if (!is_dir($user_folder)) {
    mkdir($user_folder, 0777, true);
}

// Generate full target path
$file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
$target_path = $user_folder . $chunk_index . '.' . $file_extension;

// Move uploaded file
if (!move_uploaded_file($file['tmp_name'], $target_path)) {
    $err_msg = 'Failed to move uploaded file. ';
    if (!is_writable($user_folder))
        $err_msg .= 'Directory not writable. ';
    if (!is_uploaded_file($file['tmp_name']))
        $err_msg .= 'File not a valid uploaded file. ';
    $err_msg .= 'Target path: ' . $target_path;
    error_log("Chunk Upload Error: " . $err_msg);
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $err_msg]);
    exit;
}

// --- 4. Save chunk metadata in DB ---
$sql_insert = "INSERT INTO exam_video_chunks (session_id, chunk_index, file_path, upload_time) 
               VALUES (?, ?, ?, CURRENT_TIMESTAMP())
               ON DUPLICATE KEY UPDATE file_path = VALUES(file_path), upload_time = CURRENT_TIMESTAMP()";

$insert_id = execute_non_query($mysqli, $sql_insert, 'iis', [$session_id, $chunk_index, $target_path]);

if ($insert_id === false) {
    unlink($target_path); // Remove file if DB insert failed
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Failed to save chunk metadata to database.']);
    exit;
}

// --- 5. Success response ---
echo json_encode([
    'success' => true,
    'message' => 'Chunk uploaded successfully.',
    'session_id' => $session_id,
    'chunk_index' => $chunk_index,
    'file_path' => $target_path
]);

$mysqli->close();
?>