<?php
// grade_theory.php - Admin Manual Grading Interface for Theory Questions

require_once '../db.php';
require_admin();

$page_title = 'Manual Theory Grading';
$session_id = (int) ($_GET['session_id'] ?? 0);

if ($session_id === 0) {
    $_SESSION['error_message'] = "Invalid exam session ID.";
    redirect('admin/test_takers.php');
}

// 1. Fetch Session Details
$sql_session = "SELECT S.session_id, U.full_name, E.exam_title, S.total_score, S.is_submitted
                FROM student_exam_sessions S
                JOIN users U ON S.student_id = U.user_id
                JOIN exams E ON S.exam_id = E.exam_id
                WHERE S.session_id = ?";
$result_session = execute_query($mysqli, $sql_session, 'i', [$session_id]);

if (!$result_session || $result_session->num_rows === 0) {
    $_SESSION['error_message'] = "Exam session not found.";
    redirect('admin/test_takers.php');
}
$session_data = $result_session->fetch_assoc();

if (!$session_data['is_submitted']) {
    $_SESSION['error_message'] = "This exam has not been submitted yet.";
    redirect('admin/test_takers.php');
}

// 2. Handle POST Request (Save Grades)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $mysqli->begin_transaction();
    $total_score_update = 0.00;
    $success = true;

    try {
        // Update individual theory question scores
        $sql_update_answer = "UPDATE student_answers SET score_earned = ?, is_correct = ? WHERE session_id = ? AND question_id = ?";

        foreach ($_POST['score'] as $question_id => $score_earned) {
            $question_id = (int) $question_id;
            $score_earned = (float) $score_earned;
            $is_correct = $score_earned > 0 ? 1 : 0; // Simple logic: if score > 0, mark as correct

            // Fetch max points for validation
            $sql_max_points = "SELECT points FROM exam_questions WHERE question_id = ?";
            $result_max = execute_query($mysqli, $sql_max_points, 'i', [$question_id]);
            $max_points = $result_max ? (float) $result_max->fetch_assoc()['points'] : 0.00;

            // Basic validation: score cannot exceed max points
            if ($score_earned > $max_points) {
                $score_earned = $max_points;
            }

            // Execute update
            $update_result = execute_non_query($mysqli, $sql_update_answer, 'diii', [$score_earned, $is_correct, $session_id, $question_id]);

            if ($update_result === false) {
                throw new Exception("Failed to update score for question ID: $question_id");
            }
        }

        // Recalculate total score for the session
        $sql_recalculate = "SELECT SUM(score_earned) AS new_total FROM student_answers WHERE session_id = ?";
        $result_recalculate = execute_query($mysqli, $sql_recalculate, 'i', [$session_id]);
        $new_total_score = $result_recalculate ? (float) $result_recalculate->fetch_assoc()['new_total'] : 0.00;

        // Update session total score
        $sql_update_session = "UPDATE student_exam_sessions SET total_score = ? WHERE session_id = ?";
        $update_session_result = execute_non_query($mysqli, $sql_update_session, 'di', [$new_total_score, $session_id]);

        if ($update_session_result === false) {
            throw new Exception("Failed to update session total score.");
        }

        $mysqli->commit();
        $_SESSION['success_message'] = "Theory questions graded and session total score updated successfully. New Total Score: " . number_format($new_total_score, 2);
        redirect('admin/test_takers.php');

    } catch (Exception $e) {
        $mysqli->rollback();
        $_SESSION['error_message'] = "Grading failed: " . $e->getMessage();
        // Fall through to display the form again with an error message
    }
}

// 3. Fetch Theory Questions and Student Answers
$sql_questions = "SELECT 
                    Q.question_id, 
                    Q.question_text, 
                    Q.points, 
                    A.student_answer, 
                    A.score_earned
                  FROM student_answers A
                  JOIN exam_questions Q ON A.question_id = Q.question_id
                  WHERE A.session_id = ? AND Q.question_type = 'Theory'";
$result_questions = execute_query($mysqli, $sql_questions, 'i', [$session_id]);
$theory_questions = $result_questions ? $result_questions->fetch_all(MYSQLI_ASSOC) : [];

// 4. Fetch Total Possible Score for the Exam
$sql_total_possible = "SELECT SUM(points) as total_possible FROM exam_questions WHERE exam_id = (SELECT exam_id FROM student_exam_sessions WHERE session_id = ?)";
$result_total = execute_query($mysqli, $sql_total_possible, 'i', [$session_id]);
$total_possible = $result_total ? (float) $result_total->fetch_assoc()['total_possible'] : 0.00;


include 'includes/header.php';
?>

<div class="admin-content">
    <h2>Manual Grading: <?php echo htmlspecialchars($session_data['exam_title']); ?></h2>
    <h3>Student: <?php echo htmlspecialchars($session_data['full_name']); ?></h3>
    <p>Current Total Score: <strong><?php echo number_format($session_data['total_score'], 2); ?> /
            <?php echo number_format($total_possible, 2); ?></strong></p>
    <p class="text-center"><a href="test_takers.php">← Back to Test Takers List</a></p>

    <?php if (empty($theory_questions)): ?>
        <p class="success">No Theory questions found for this exam session, or all questions have been auto-graded.</p>
    <?php else: ?>
        <form method="POST" action="grade_theory.php?session_id=<?php echo $session_id; ?>">
            <?php foreach ($theory_questions as $q): ?>
                <div class="grading-item"
                    style="border: 1px solid #ccc; padding: 15px; margin-bottom: 20px; border-radius: 5px;">
                    <h4>Question ID: <?php echo $q['question_id']; ?> (Max Points:
                        <?php echo number_format($q['points'], 2); ?>)
                    </h4>
                    <p><strong>Question:</strong> <?php echo nl2br(htmlspecialchars($q['question_text'])); ?></p>
                    <p><strong>Student Answer:</strong></p>
                    <div style="border: 1px solid #eee; padding: 10px; background-color: #f9f9f9; white-space: pre-wrap;">
                        <?php echo htmlspecialchars($q['student_answer']); ?>
                    </div>

                    <!-- <div style="margin-top: 10px;">
                        <button type="button" class="btn-ai-grade" data-session-id="<?php echo $session_id; ?>" data-question-id="<?php echo $q['question_id']; ?>" style="background-color: #007bff; color: white; border: none; padding: 5px 10px; cursor: pointer; border-radius: 3px;">AI Grade</button>
                        <span class="ai-reason-<?php echo $q['question_id']; ?>" style="margin-left                        <?php echo htmlspecialchars($q['student_answer']); ?>
                    </div>
                    
                    <div style="margin-top: 10px;">
                        <button type="button" class="btn-ai-grade" data-session-id="<?php echo $session_id; ?>" data-question-id="<?php echo $q['question_id']; ?>" style="background-color: #007bff; color: white; border: none; padding: 5px 10px; cursor: pointer; border-radius: 3px;">AI Grade</button>
                        <span class="ai-reason-<?php echo $q['question_id']; ?>" style="margin-left: 10px; font-style: italic; color: #555;"></span>
                    </div> -->

                    <div style="margin-top: 10px;">
                        <label for="score_<?php echo $q['question_id']; ?>">Score Earned:</label> <input type="number"
                            name="score[<?php echo $q['question_id']; ?>]" id="score_<?php echo $q['question_id']; ?>"
                            value="<?php echo number_format($q['score_earned'], 2, '.', ''); ?>" min="0"
                            max="<?php echo number_format($q['points'], 2, '.', ''); ?>" step="0.01" required>
                        / <?php echo number_format($q['points'], 2); ?>
                    </div>
                </div>
            <?php endforeach; ?>

            <button type="submit"
                style="padding: 10px 20px; background-color: #4CAF50; color: white; border: none; cursor: pointer; border-radius: 5px;">Save
                Grades and Update Total Score</button>
        </form>
    <?php endif; ?>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        const aiGradeButtons = document.querySelectorAll('.btn-ai-grade');

        aiGradeButtons.forEach(button => {
            button.addEventListener('click', function () {
                const sessionId = this.getAttribute('data-session-id');
                const questionId = this.getAttribute('data-question-id');
                const scoreInput = document.getElementById('score_' + questionId);
                const reasonSpan = document.querySelector('.ai-reason-' + questionId);
                const originalText = this.textContent;

                // Disable button and show loading
                this.disabled = true;
                this.textContent = 'Grading...';
                reasonSpan.textContent = 'Waiting for AI...';

                fetch('ai_grade.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'session_id=' + sessionId + '&question_id=' + questionId
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // Update the score input field
                            scoreInput.value = data.new_score;

                            // Display the AI's reason
                            reasonSpan.textContent = 'AI Reason: ' + data.reason;
                            reasonSpan.style.color = '#28a745'; // Green for success

                            // Alert the user to save the form
                            alert('AI Grading complete. Remember to click "Save Grades" to finalize the session score.');
                        } else {
                            reasonSpan.textContent = 'AI Grading Failed: ' + data.message;
                            reasonSpan.style.color = '#dc3545'; // Red for error
                            alert('AI Grading Failed: ' + data.message);
                        }
                    })
                    .catch(error => {
                        reasonSpan.textContent = 'Network Error: Could not connect to server.';
                        reasonSpan.style.color = '#dc3545';
                        console.error('AI Grade Error:', error);
                    })
                    .finally(() => {
                        // Re-enable button
                        this.disabled = false;
                        this.textContent = originalText;
                    });
            });
        });
    });
</script>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        const aiGradeButtons = document.querySelectorAll('.btn-ai-grade');

        aiGradeButtons.forEach(button => {
            button.addEventListener('click', function () {
                const sessionId = this.getAttribute('data-session-id');
                const questionId = this.getAttribute('data-question-id');
                const scoreInput = document.getElementById('score_' + questionId);
                const reasonSpan = document.querySelector('.ai-reason-' + questionId);
                const originalText = this.textContent;

                // Disable button and show loading
                this.disabled = true;
                this.textContent = 'Grading...';
                reasonSpan.textContent = 'Waiting for AI...';

                fetch('ai_grade.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'session_id=' + sessionId + '&question_id=' + questionId
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // Update the score input field
                            scoreInput.value = data.new_score;

                            // Display the AI's reason
                            reasonSpan.textContent = 'AI Reason: ' + data.reason;
                            reasonSpan.style.color = '#28a745'; // Green for success

                            // Alert the user to save the form
                            alert('AI Grading complete. Remember to click "Save Grades" to finalize the session score.');
                        } else {
                            reasonSpan.textContent = 'AI Grading Failed: ' + data.message;
                            reasonSpan.style.color = '#dc3545'; // Red for error
                            alert('AI Grading Failed: ' + data.message);
                        }
                    })
                    .catch(error => {
                        reasonSpan.textContent = 'Network Error: Could not connect to server.';
                        reasonSpan.style.color = '#dc3545';
                        console.error('AI Grade Error:', error);
                    })
                    .finally(() => {
                        // Re-enable button
                        this.disabled = false;
                        this.textContent = originalText;
                    });
            });
        });
    });
</script>

<?php
include 'includes/footer.php';
?>