<?php
// admin/test_takers.php - Admin List of Test Takers and Exam Status

require_once '../db.php';
require_admin();

// Set page title for header
$page_title = 'Test-Takers List & Status';

// Fetch all test-takers (user_type = 'Student')
$sql_users = "SELECT user_id, full_name, username, email, grade FROM users WHERE user_type = 'Student' ORDER BY full_name";
$users_result = execute_query($mysqli, $sql_users);
$users = $users_result ? $users_result->fetch_all(MYSQLI_ASSOC) : [];

// Fetch all exam sessions for status lookup
// This query is complex as it needs to join users, sessions, and exams
$sql_sessions = "
    SELECT 
        SES.student_id,
        E.exam_title,
        SES.start_time,
        SES.end_time,
        SES.total_score AS score,
        SES.proctoring_video_path,
        SES.video_upload_complete,
        SES.session_id,
        E.duration_minutes
    FROM 
        student_exam_sessions SES
    JOIN 
        exams E ON SES.exam_id = E.exam_id
    ORDER BY 
        SES.start_time DESC
";
$sessions_result = execute_query($mysqli, $sql_sessions);
$sessions = $sessions_result ? $sessions_result->fetch_all(MYSQLI_ASSOC) : [];

// Organize sessions by student_id for easy lookup
$user_sessions = [];
foreach ($sessions as $session) {
    $user_sessions[$session['student_id']][] = $session;
}

/**
 * Formats the time spent on an exam.
 * @param string $start_time
 * @param string|null $end_time
 * @return string
 */
function format_time_spent($start_time, $end_time)
{
    if (!$start_time) {
        return 'N/A';
    }
    
    // CRITICAL FIX: Ensure start_time is valid before processing
    $start = strtotime($start_time);
    if ($start === false) {
        return 'Invalid Start Time';
    }
    
    // If end_time is null, the exam is ongoing, so use current time
    $end = $end_time ? strtotime($end_time) : time();
       // If end time is before start time, it's an invalid record. Return N/A gracefully.
    if ($end < $start) {
        return 'N/A (Corrupted Record)';
    }    
    $time_diff = $end - $start;
    
    $hours = floor($time_diff / 3600);
    $minutes = floor(($time_diff % 3600) / 60);
    $seconds = $time_diff % 60;

    if ($hours > 0) {
        return sprintf("%d hr %d min %d sec", $hours, $minutes, $seconds);
    } elseif ($minutes > 0) {
        return sprintf("%d min %d sec", $minutes, $seconds);
    } else {
        return sprintf("%d sec", $seconds);
    }
}

include 'includes/header.php';
?>

<div class="admin-content">
    <h2>Test-Takers List</h2>

    <?php if (empty($users)): ?>
        <p>No test-takers found in the system.</p>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>S/n</th>
                    <th>Full Name</th>
                    <th>Staff ID/Username</th>
                    <th>Email</th>
                    <th>Grade</th>
                    <th>Exam Status</th>
                    <th>Score</th>
                    <th>Time Spent</th>
                    <th>Proctoring Video</th>
                </tr>
            </thead>
            <tbody>
                <?php $i = 1;
                foreach ($users as $user): ?>
                    <tr>
                        <td><?= $i++; ?></td>
                        <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                        <td><?php echo htmlspecialchars($user['username']); ?></td>
                        <td><?php echo htmlspecialchars($user['email']); ?></td>
                        <td><?php echo htmlspecialchars($user['grade']); ?></td>
                        <?php
                        $status_html = '';
                        $score_html = '';
                        $time_html = '';

                        if (isset($user_sessions[$user['user_id']])) {
                            $latest_session = $user_sessions[$user['user_id']][0]; // Assuming ordered by start_time DESC
                
                            $exam_title = htmlspecialchars($latest_session['exam_title']);
                            $score = $latest_session['score'];
                            $session_id_for_calc = $latest_session['session_id'];
                            
                            // Calculate total possible score based on questions answered
                            $sql_total_possible = "SELECT SUM(Q.points) as total_possible 
                                                   FROM student_answers SA 
                                                   JOIN exam_questions Q ON SA.question_id = Q.question_id 
                                                   WHERE SA.session_id = ?";
                            $result_total = execute_query($mysqli, $sql_total_possible, 'i', [$session_id_for_calc]);
                            $total_possible = 0;
                            if ($result_total && $result_total->num_rows > 0) {
                                $total_row = $result_total->fetch_assoc();
                                $total_possible = (float)($total_row['total_possible'] ?? 0);
                            }

                            if ($latest_session['end_time']) {
                                // Completed Exam
                                $status_html = "<span class='status-completed'>Completed: $exam_title</span>";
                                // Add link to manual grading page
                                $status_html .= " <a href='grade_theory.php?session_id=" . $latest_session['session_id'] . "' class='btn btn-primary btn-small'>Grade Theory</a>";
                                // CRITICAL FIX: Ensure score is displayed correctly. The score column in the session table
                                // is likely a percentage or a raw score. Assuming it's a raw score for now.
                                // The original request was to fix scoring based on allocated points.
                                // The score in the session table is total_score, which should be the sum of score_earned.
                                // The total_possible calculation is correct.
                                $score_html = sprintf("%.2f / %.2f", $score, $total_possible);
                                $time_html = format_time_spent($latest_session['start_time'], $latest_session['end_time']);
                        } else {
                            // In Progress Exam
                            $status_html = "<span class='status-in-progress'>In Progress: $exam_title</span>";
                            $score_html = 'N/A';
                            // Add data attribute with start time for JavaScript to calculate elapsed time
                            $start_timestamp = strtotime($latest_session['start_time']);
                            $duration_seconds = $latest_session['duration_minutes'] * 60;
$time_html = "<span class='time-counter' data-start-time='$start_timestamp' data-duration='$duration_seconds'>" . format_time_spent($latest_session['start_time'], null) . "</span>";
                        }

                            // Video Processing Link
                            $video_link = '';
                            if ($latest_session['proctoring_video_path']) {
                                // Final video exists (automatically processed)
                                $video_link = "<a href='" . BASE_URL . htmlspecialchars($latest_session['proctoring_video_path']) . "' target='_blank' class='btn btn-success'>View Video</a>";
                            } elseif ($latest_session['video_upload_complete']) {
                                // Video chunks uploaded and marked complete, processing should be automatic
                                // Check if video file exists but database not updated
                                $video_link = "<span class='status-processing'>Processing...</span> <a href='process_video.php?session_id=" . $latest_session['session_id'] . "' class='btn btn-warning btn-small'>Retry</a>";
                            } elseif ($latest_session['end_time']) {
                                // Exam submitted, but video not yet finalized
                                $video_link = "<span class='status-pending'>Upload in Progress</span>";
                            } else {
                                // Video chunks still uploading or exam in progress
                                $video_link = "<span class='status-pending'>Video Pending</span>";
                            }

                        } else {
                            // No Exam Taken
                            $status_html = "<span class='status-pending'>Pending Exam</span>";
                            $score_html = 'N/A';
                            $time_html = 'N/A';
                            $video_link = 'N/A';
                        }
                        ?>
                        <td><?php echo $status_html; ?></td>
                        <td><?php echo $score_html; ?></td>
                        <td><?php echo $time_html; ?></td>
                        <td><?php echo $video_link; ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<script>
// Auto-update time counter for ongoing exams
function updateTimeCounters() {
    const counters = document.querySelectorAll('.time-counter');
    
    counters.forEach(counter => {
        const startTime = parseInt(counter.getAttribute('data-start-time'));
        if (!startTime) return;
        
        const now = Math.floor(Date.now() / 1000); // Current time in seconds
        const elapsed = now - startTime;
        const duration = parseInt(counter.getAttribute('data-duration'));
        
        let displayTime = elapsed;
        if (!isNaN(duration) && elapsed > duration) {
            displayTime = duration;
        }
        
        if (elapsed < 0) {
            counter.textContent = 'N/A (Invalid Time)';
            return;
        }
        
        const hours = Math.floor(displayTime / 3600);
        const minutes = Math.floor((displayTime % 3600) / 60);
        const seconds = displayTime % 60;
        
        let timeString = '';
        if (hours > 0) {
            timeString = `${hours} hr ${minutes} min ${seconds} sec`;
        } else if (minutes > 0) {
            timeString = `${minutes} min ${seconds} sec`;
        } else {
            timeString = `${seconds} sec`;
        }
        
        counter.textContent = timeString;
    });
}

// Update counters every second
setInterval(updateTimeCounters, 1000);

// Initial update on page load
updateTimeCounters();
</script>

<?php
include 'includes/footer.php';
?>