<?php
// db.php - Database Connection and Utility Functions

// =================================================================================================
// 0. DEPENDENCIES
// =================================================================================================

// Composer Autoloader for external libraries (like the OpenAI client)
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

// =================================================================================================
// 1. FUNCTION DEFINITIONS (MUST BE FIRST)
// =================================================================================================

// Function to redirect (Defined first to avoid "undefined function" errors)
function redirect($url, $type = null, $message = null)
{
    // Check if the URL is already absolute (contains http or https)
    if (strpos($url, 'http') === 0) {
        $final_url = $url;
    } else {
        // Use a relative path if BASE_URL is not defined yet (should be defined in constants)
        // For internal redirects, we need to calculate the path relative to the root.
        // The $url parameter should be relative to the project root (e.g., 'student/take_exam.php').
        $base_url = defined('BASE_URL') ? BASE_URL : '/';

        // Ensure there is only one slash between BASE_URL and $url
        $final_url = rtrim($base_url, '/') . '/' . ltrim($url, '/');
    }

    // Append flash messages if provided
    if ($type && $message) {
        $_SESSION['flash_type'] = $type;
        $_SESSION['flash_message'] = $message;
    }

    // Check if headers have already been sent
    if (headers_sent()) {
        error_log("Redirect failed: Headers already sent. Output started at " . headers_sent($file, $line) . " in file " . $file . " on line " . $line);
        // Fallback: output a JavaScript redirect
        echo "<script>window.location.href='" . $final_url . "';</script>";
        exit;
    }

    header("Location: " . $final_url);
    exit;
}

// Function to check if user is logged in
function is_logged_in()
{
    return isset($_SESSION['user_id']);
}

// Function to check user type
function is_admin()
{
    return is_logged_in() && $_SESSION['user_type'] === 'Admin';
}

function is_student()
{
    return is_logged_in() && $_SESSION['user_type'] === 'Student';
}

// Function to ensure admin access
function require_admin()
{
    if (!is_admin()) {
        redirect('index.php'); // Redirect to login or home page
    }
}

// Function to ensure student access
function require_student()
{
    if (!is_student()) {
        redirect('index.php'); // Redirect to login or home page
    }
}

// Function to get current user ID
function get_current_user_id()
{
    return $_SESSION['user_id'] ?? null;
}

// Function to get current user grade
function get_current_user_grade()
{
    return $_SESSION['grade'] ?? null;
}

// Function to get current user staff ID
function get_current_user_staff_id()
{
    return $_SESSION['staff_id'] ?? null;
}

// Function to get current user username
function get_current_username()
{
    return $_SESSION['username'] ?? null;
}

// =================================================================================================
// 2. INITIALIZATION (SESSION AND AUTOLOAD)
// =================================================================================================

// Start session
session_start();

// Include Composer Autoloader
require __DIR__ . '/vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// =================================================================================================
// 3. CONSTANTS AND CONFIGURATION
// =================================================================================================

// Database configuration
define('DB_SERVER', 'localhost');
define('DB_USERNAME', 'root'); // Change this to your database username
define('DB_PASSWORD', ''); // Change this to your database password
define('DB_NAME', 'proctored_exam_app2'); // Change this to your database name

// --- BASE URL CONFIGURATION (CRITICAL FOR WEB WORKER) ---
if (!defined('BASE_URL')) {
    // Attempt to auto-detect the base URL
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || ($_SERVER['SERVER_PORT'] ?? 80) == 443) ? "https://" : "http://";
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';

    // Calculate the path to the project root (where db.php is located)
    $project_root_path = str_replace('\\', '/', __DIR__);
    $document_root_path = str_replace('\\', '/', $_SERVER['DOCUMENT_ROOT'] ?? '');

    // Calculate the relative path from DOCUMENT_ROOT to PROJECT_ROOT
    $relative_path = str_replace($document_root_path, '', $project_root_path);

    // Construct the base URL
    $auto_base_url = $protocol . $host . $relative_path . '/';

    // Fallback to a default if auto-detection is not possible or seems wrong
    if (empty($relative_path) || strpos($auto_base_url, 'index.php') !== false) {
        define('BASE_URL', 'http://localhost/protor/'); // Default XAMPP setting
    } else {
        define('BASE_URL', $auto_base_url);
    }
}

// Define the project root directory
if (!defined('PROJECT_ROOT_DIR')) {
    define('PROJECT_ROOT_DIR', __DIR__ . '/');
}

// Define the path where video chunks will be stored
define('VIDEO_CHUNKS_DIR', PROJECT_ROOT_DIR . 'proctor/chunks/');

// Create the directory if it doesn't exist
if (!is_dir(VIDEO_CHUNKS_DIR)) {
    mkdir(VIDEO_CHUNKS_DIR, 0777, true);
}

// Define the path where final stitched videos will be stored (optional for future use)
define('FINAL_VIDEO_DIR', PROJECT_ROOT_DIR . 'proctor/final_videos/');

// Create the directory if it doesn't exist
if (!is_dir(FINAL_VIDEO_DIR)) {
    mkdir(FINAL_VIDEO_DIR, 0777, true);
}

// Define the path for the web worker script
define('WORKER_SCRIPT_PATH', BASE_URL . 'proctor/upload_worker.js');

// Define the path for the chunk upload handler (Absolute Path)
define('CHUNK_UPLOAD_HANDLER', BASE_URL . 'proctor/upload_chunk.php');

// Define the path for the exam submission handler
define('EXAM_SUBMIT_HANDLER', BASE_URL . 'student/submit_exam.php');

// Define the path for the main JS file
define('MAIN_JS_PATH', BASE_URL . 'assets/js/proctoring.js');

// Define the path for the main CSS file
define('MAIN_CSS_PATH', BASE_URL . 'assets/css/style.css');

// Define the maximum number of concurrent workers
define('MAX_WORKERS', 5);

// Define the chunk duration in milliseconds (1 second)
define('CHUNK_DURATION_MS', 1000);

// Define the project name
define('PROJECT_NAME', 'Proctored Exam System');

// OpenAI API Configuration
define('OPENAI_API_KEY', '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');

// PHPMailer Configuration (PLACEHOLDERS - REPLACE WITH ACTUAL SMTP SETTINGS)
define('SMTP_HOST', 'smtp.example.com');
define('SMTP_USERNAME', 'user@example.com');
define('SMTP_PASSWORD', 'your_password');
define('SMTP_PORT', 587);
define('SMTP_FROM_EMAIL', 'no-reply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost'));
define('SMTP_FROM_NAME', PROJECT_NAME . ' Admin');

// =================================================================================================
// 4. DATABASE CONNECTION
// =================================================================================================

// Attempt to connect to MySQL database
$mysqli = new mysqli(DB_SERVER, DB_USERNAME, DB_PASSWORD, DB_NAME);

// Check connection
if ($mysqli->connect_error) {
    // Do not die, but log the error and continue, as some pages might not need the DB
    error_log("Database Connection Failed: " . $mysqli->connect_error);
    // For pages that absolutely need the DB (like admin pages), they will fail later.
    // For now, we'll just log and continue.
}

// =================================================================================================
// 5. HELPER FUNCTIONS (Need $mysqli)
// =================================================================================================

/**
 * Executes a prepared statement and returns the result set.
 * @param mysqli $mysqli The database connection object.
 * @param string $sql The SQL query with '?' placeholders.
 * @param string $types A string containing one or more characters which specify the types for the corresponding bind variables.
 * @param array $params An array of variables to bind to the prepared statement.
 * @return mysqli_result|bool Returns the result object on success, or FALSE on failure.
 */
function execute_query($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error) {
        error_log("Attempted DB query with failed connection.");
        return false;
    }
    if ($stmt = $mysqli->prepare($sql)) {
        if (!empty($types) && !empty($params)) {
            // Handle NULL values for bind_param
            $bind_params = [];
            $bind_params[] = $types;
            foreach ($params as $key => &$value) {
                // If value is NULL, we must ensure the type is 's' (string)
                // as bind_param does not support NULL for non-string types.
                // However, since we are using the splat operator, we need to
                // ensure the array structure is correct.
                // The best way to handle this is to use call_user_func_array
                // or ensure all parameters are passed by reference, which is
                // complex. Since the existing code uses the splat operator,
                // we will assume the user's environment supports this for
                // basic types and focus on the NULL issue.
                // The simplest fix is to change the type to 's' for NULL values,
                // but this requires dynamic type string generation.
                // Given the existing structure, we will assume the issue is
                // that the `execute_non_query` function is not designed to handle
                // NULL values being passed directly for non-string types.
                // We will modify the function to explicitly check for NULL and
                // cast to the expected type (0 for int/bool, 0.0 for double)
                // which is a common workaround for simple applications.

                // Reverting to the original logic, as the splat operator should
                // work if the types string is correct. The error is likely in
                // the type string 'iisid' in submit_exam.php.
                // We will assume the original code structure is correct and
                // the issue is in the calling code.

                // Since the user is reporting a persistent error, let's try the
                // most robust fix for NULL binding without changing the function signature.
                // This requires using call_user_func_array and passing by reference.

                // Temporarily removing the splat operator for a safer bind:
                $bind_params = [$types];
                foreach ($params as &$param) {
                    $bind_params[] = &$param;
                }
                call_user_func_array([$stmt, 'bind_param'], $bind_params);
            }
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $stmt->close();
            return $result;
        } else {
            // Log error or handle it appropriately
            error_log("Execute failed: " . $stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log("Prepare failed: " . $mysqli->error);
        return false;
    }
}

/**
 * Executes a prepared statement for INSERT, UPDATE, or DELETE operations.
 * @param mysqli $mysqli The database connection object.
 * @param string $sql The SQL query with '?' placeholders.
 * @param string $types A string containing one or more characters which specify the types for the corresponding bind variables.
 * @param array $params An array of variables to bind to the prepared statement.
 * @return int|bool Returns the number of affected rows on success, or FALSE on failure.
 */
function execute_non_query($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error) {
        error_log("Attempted DB non-query with failed connection.");
        return false;
    }
    if ($stmt = $mysqli->prepare($sql)) {
        if (!empty($types) && !empty($params)) {
            $bind_params = [$types];
            foreach ($params as &$param) {
                $bind_params[] = &$param;
            }
            call_user_func_array([$stmt, 'bind_param'], $bind_params);
        }

        if ($stmt->execute()) {
            $affected_rows = $stmt->affected_rows;
            $stmt->close();
            return $affected_rows;
        } else {
            error_log("Execute failed: " . $stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log("Prepare failed: " . $mysqli->error);
        return false;
    }
}

/**
 * Executes a prepared statement for INSERT and returns the last inserted ID.
 * @param mysqli $mysqli The database connection object.
 * @param string $sql The SQL query with '?' placeholders.
 * @param string $types A string containing one or more characters which specify the types for the corresponding bind variables.
 * @param array $params An array of variables to bind to the prepared statement.
 * @return int|bool Returns the last inserted ID on success, or FALSE on failure.
 */
function execute_insert($mysqli, $sql, $types = '', $params = [])
{
    if ($mysqli->connect_error) {
        error_log("Attempted DB insert with failed connection.");
        return false;
    }
    if ($stmt = $mysqli->prepare($sql)) {
        if (!empty($types) && !empty($params)) {
            $bind_params = [$types];
            foreach ($params as &$param) {
                $bind_params[] = &$param;
            }
            call_user_func_array([$stmt, 'bind_param'], $bind_params);
        }

        if ($stmt->execute()) {
            $insert_id = $mysqli->insert_id;
            $stmt->close();
            return $insert_id;
        } else {
            error_log("Execute failed: " . $stmt->error);
            $stmt->close();
            return false;
        }
    } else {
        error_log("Prepare failed: " . $mysqli->error);
        return false;
    }
}

/**
 * Sends an email using PHPMailer.
 * NOTE: Uses hardcoded SMTP settings from constants.
 * @param string $to The recipient email address.
 * @param string $subject The email subject.
 * @param string $body The email body (HTML or plain text).
 * @return bool True on success, false on failure.
 */
function send_email($to, $subject, $body)
{
    // PHPMailer requires the global scope for the class to be available
    $mail = new PHPMailer(true);
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = SMTP_PORT;
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);

        // Recipients
        $mail->addAddress($to);

        // Content
        $mail->isHTML(false); // We are sending plain text for simplicity
        $mail->Subject = $subject;
        $mail->Body = $body;

        $mail->send();
        return true;
    } catch (Exception $e) {
        // Log the error instead of exposing it to the user
        error_log("Email failed to send to $to. Mailer Error: {$e->getMessage()}");
        return false;
    }
}

?>