<?php
// ai_grade.php - AJAX endpoint to trigger AI grading for a single theory question

header('Content-Type: application/json');

require_once '../db.php';
require_once '../ai_utils.php'; // Include the AI utility functions
require_admin();

$response = ['success' => false, 'message' => 'Invalid request.', 'new_score' => 0.00, 'reason' => ''];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode($response);
    exit;
}

$session_id = (int)($_POST['session_id'] ?? 0);
$question_id = (int)($_POST['question_id'] ?? 0);

if ($session_id === 0 || $question_id === 0) {
    $response['message'] = 'Missing session ID or question ID.';
    echo json_encode($response);
    exit;
}

// 1. Fetch Question and Answer Data
$sql_data = "SELECT 
                Q.question_text, 
                Q.correct_answer, 
                Q.points, 
                A.student_answer
             FROM student_answers A
             JOIN exam_questions Q ON A.question_id = Q.question_id
             WHERE A.session_id = ? AND A.question_id = ? AND Q.question_type = 'Theory'";
$result_data = execute_query($mysqli, $sql_data, 'ii', [$session_id, $question_id]);

if (!$result_data || $result_data->num_rows === 0) {
    $response['message'] = 'Theory question data not found for this session.';
    echo json_encode($response);
    exit;
}
$data = $result_data->fetch_assoc();

$question_text = $data['question_text'];
$correct_answer = $data['correct_answer']; // Using correct_answer for the model answer
$student_answer = $data['student_answer'];
$max_points = (float)$data['points'];

// 2. Call AI Grading Function
$grading_result = ai_grade_theory_answer($question_text, $correct_answer, $student_answer, $max_points);

if ($grading_result === null) {
    $response['message'] = 'AI Grading failed. Check server logs for details.';
    echo json_encode($response);
    exit;
}

$new_score = $grading_result['score'];
$reason = $grading_result['reason'];
$is_correct = $new_score > 0 ? 1 : 0; // Simple logic: if score > 0, mark as correct

// 3. Update Database
$mysqli->begin_transaction();

try {
    // Update individual theory question score
    $sql_update_answer = "UPDATE student_answers SET score_earned = ?, is_correct = ? WHERE session_id = ? AND question_id = ?";
    $update_result = execute_non_query($mysqli, $sql_update_answer, 'diii', [$new_score, $is_correct, $session_id, $question_id]);

    if ($update_result === false) {
        throw new Exception("Failed to update score for question ID: $question_id");
    }

    // Recalculate total score for the session
    $sql_recalculate = "SELECT SUM(score_earned) AS new_total FROM student_answers WHERE session_id = ?";
    $result_recalculate = execute_query($mysqli, $sql_recalculate, 'i', [$session_id]);
    $new_total_score = $result_recalculate ? (float)$result_recalculate->fetch_assoc()['new_total'] : 0.00;

    // Update session total score
    $sql_update_session = "UPDATE student_exam_sessions SET total_score = ? WHERE session_id = ?";
    $update_session_result = execute_non_query($mysqli, $sql_update_session, 'di', [$new_total_score, $session_id]);

    if ($update_session_result === false) {
        throw new Exception("Failed to update session total score.");
    }

    $mysqli->commit();

    // 4. Success Response
    $response['success'] = true;
    $response['message'] = 'AI Grading complete. Score updated.';
    $response['new_score'] = number_format($new_score, 2, '.', '');
    $response['reason'] = $reason;

} catch (Exception $e) {
    $mysqli->rollback();
    $response['message'] = 'Database update failed: ' . $e->getMessage();
    error_log($response['message']);
}

$mysqli->close();
echo json_encode($response);
?>
