<?php
// questions.php - Admin Question Bank Management

require_once '../db.php';
require_admin();

// Set page title for header
$page_title = 'Question Bank Management';

$error = '';
$success = '';
$question_types = ['MCQ', 'Theory', 'Fill-in-the-gaps']; // Question types are fixed, but the available types for a new question will be filtered by the selected exam's mode

// Fetch all exams for the dropdown, including their exam_mode
$sql_exams = "SELECT exam_id, exam_title, exam_mode FROM exams ORDER BY exam_title";
$exams_result = execute_query($mysqli, $sql_exams);
$exams = $exams_result ? $exams_result->fetch_all(MYSQLI_ASSOC) : [];

// Convert exams array to a map for easy lookup in JavaScript
$exams_map = [];
foreach ($exams as $exam) {
    $exams_map[$exam['exam_id']] = $exam['exam_mode'];
}

// Handle Delete Request
if (isset($_GET['action']) && $_GET['action'] === 'delete') {
    $question_id_to_delete = (int) ($_GET['id'] ?? 0);

    if ($question_id_to_delete > 0) {
        // Check if the question is part of any active exam session
        $sql_check_session = "SELECT COUNT(*) FROM student_answers WHERE question_id = ?";
        $result_check = execute_query($mysqli, $sql_check_session, 'i', [$question_id_to_delete]);
        $count = $result_check ? $result_check->fetch_row()[0] : 0;

        if ($count > 0) {
            $error = "Cannot delete question ID {$question_id_to_delete}. It is linked to {$count} student answers.";
        } else {
            // Delete the question
            $sql_delete = "DELETE FROM exam_questions WHERE question_id = ?";
            $affected_rows = execute_non_query($mysqli, $sql_delete, 'i', [$question_id_to_delete]);

            if ($affected_rows !== false && $affected_rows > 0) {
                $success = "Question ID {$question_id_to_delete} deleted successfully.";
            } else {
                $db_error = $mysqli->error;
                error_log("Failed to delete question ID {$question_id_to_delete}. DB Error: {$db_error}");
                $error = "Error deleting question ID {$question_id_to_delete}. Database error occurred.";
            }
        }
    } else {
        $error = "Invalid question ID for deletion.";
    }
    // Redirect to clear the GET parameters
    redirect('questions.php', $error ? 'error' : 'success', $error ?: $success);
}

// Handle Add Question Request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exam_id = (int) ($_POST['exam_id'] ?? 0);
    $question_type = $_POST['question_type'] ?? '';
    $question_text = trim($_POST['question_text'] ?? '');
    $points = (float) ($_POST['points'] ?? 1.00);

    // Fetch the selected exam's mode for validation
    $selected_exam_mode = $exams_map[$exam_id] ?? null;

    // Validation
    if (empty($question_text) || $exam_id <= 0 || !in_array($question_type, $question_types)) {
        $error = 'Please fill in the question text, select an exam, and a valid question type.';
    } elseif ($selected_exam_mode && $selected_exam_mode !== 'Mixed' && $selected_exam_mode !== $question_type) {
        $error = "The selected exam is in '{$selected_exam_mode}' mode and only allows '{$selected_exam_mode}' questions.";
    } else {
        $option_a = $_POST['option_a'] ?? null;
        $option_b = $_POST['option_b'] ?? null;
        $option_c = $_POST['option_c'] ?? null;
        $option_d = $_POST['option_d'] ?? null;
        $correct_option = $_POST['correct_option'] ?? null;
        $correct_answer = $_POST['correct_answer'] ?? null;
        $expected_keywords = $_POST['expected_keywords'] ?? null;

        $sql_insert = "INSERT INTO exam_questions (exam_id, question_type, question_text, option_a, option_b, option_c, option_d, correct_option, correct_answer, expected_keywords, points) 
                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

        $insert_id = execute_insert($mysqli, $sql_insert, 'issssssssds', [
            $exam_id,
            $question_type,
            $question_text,
            $option_a,
            $option_b,
            $option_c,
            $option_d,
            $correct_option,
            $correct_answer,
            $expected_keywords,
            $points
        ]);

        if ($insert_id) {
            $success = "Question added successfully (ID: {$insert_id}).";
            // Clear POST data to reset form
            $_POST = [];
        } else {
            $error = 'Error adding question.';
        }
    }
}

// Fetch all questions for display
$sql_select = "SELECT Q.*, E.exam_title 
               FROM exam_questions Q
               JOIN exams E ON Q.exam_id = E.exam_id
               ORDER BY Q.question_id DESC";
$result = execute_query($mysqli, $sql_select);
$questions = $result ? $result->fetch_all(MYSQLI_ASSOC) : [];

include 'includes/header.php';
?>

<div class="mb-20">
    <a href="<?php echo BASE_URL; ?>admin/exams.php">Manage Exams</a>
</div>

<h2>Add New Question</h2>
<?php if ($error): ?>
    <p class="error"><?php echo $error; ?></p>
<?php endif; ?>
<?php if ($success): ?>
    <p class="success"><?php echo $success; ?></p>
<?php endif; ?>
<form method="POST" action="questions.php">
    <div class="form-group">
        <label for="exam_id">Assign to Exam:</label>
        <select id="exam_id" name="exam_id" onchange="updateQuestionTypes()" required>
            <option value="">Select Exam</option>
            <?php foreach ($exams as $exam): ?>
                <option value="<?php echo $exam['exam_id']; ?>"
                    data-mode="<?php echo htmlspecialchars($exam['exam_mode']); ?>" <?php echo (($_POST['exam_id'] ?? '') == $exam['exam_id']) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($exam['exam_title']); ?> (Mode:
                    <?php echo htmlspecialchars($exam['exam_mode']); ?>)
                </option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="form-group">
        <label for="question_type">Question Type:</label>
        <select id="question_type" name="question_type" onchange="toggleQuestionFields()" required>
            <option value="">Select Type</option>
            <?php foreach ($question_types as $type): ?>
                <option value="<?php echo $type; ?>" class="question-type-option" data-type="<?php echo $type; ?>" <?php echo (($_POST['question_type'] ?? '') === $type) ? 'selected' : ''; ?>>
                    <?php echo $type; ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="form-group">
        <label for="question_text">Question Text:</label>
        <textarea id="question_text" name="question_text"
            required><?php echo htmlspecialchars($_POST['question_text'] ?? ''); ?></textarea>
    </div>
    <div class="form-group">
        <label for="points">Points:</label>
        <input type="number" id="points" name="points" min="0.01" step="0.01"
            value="<?php echo htmlspecialchars($_POST['points'] ?? 1.00); ?>" required>
    </div>

    <!-- MCQ Fields -->
    <div id="mcq_fields" style="display: none; border: 1px solid #ccc; padding: 15px; margin-bottom: 15px;">
        <h4 style="margin-top: 0;">MCQ Options</h4>
        <div class="form-group">
            <label for="option_a">Option A:</label>
            <input type="text" id="option_a" name="option_a"
                value="<?php echo htmlspecialchars($_POST['option_a'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_b">Option B:</label>
            <input type="text" id="option_b" name="option_b"
                value="<?php echo htmlspecialchars($_POST['option_b'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_c">Option C:</label>
            <input type="text" id="option_c" name="option_c"
                value="<?php echo htmlspecialchars($_POST['option_c'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="option_d">Option D:</label>
            <input type="text" id="option_d" name="option_d"
                value="<?php echo htmlspecialchars($_POST['option_d'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="correct_option">Correct Option:</label>
            <select id="correct_option" name="correct_option">
                <option value="">Select Correct Option</option>
                <option value="A" <?php echo (($_POST['correct_option'] ?? '') === 'A') ? 'selected' : ''; ?>>A</option>
                <option value="B" <?php echo (($_POST['correct_option'] ?? '') === 'B') ? 'selected' : ''; ?>>B</option>
                <option value="C" <?php echo (($_POST['correct_option'] ?? '') === 'C') ? 'selected' : ''; ?>>C</option>
                <option value="D" <?php echo (($_POST['correct_option'] ?? '') === 'D') ? 'selected' : ''; ?>>D</option>
            </select>
        </div>
    </div>

    <!-- Theory/Fill-in-the-gaps Fields -->
    <div id="theory_fields" style="display: none; border: 1px solid #ccc; padding: 15px; margin-bottom: 15px;">
        <h4 style="margin-top: 0;">Answer Details</h4>
        <div class="form-group">
            <label for="correct_answer">Correct Answer (for Fill-in-the-gaps):</label>
            <input type="text" id="correct_answer" name="correct_answer"
                value="<?php echo htmlspecialchars($_POST['correct_answer'] ?? ''); ?>">
        </div>
        <div class="form-group">
            <label for="expected_keywords">Expected Keywords (for Theory - comma separated):</label>
            <textarea id="expected_keywords"
                name="expected_keywords"><?php echo htmlspecialchars($_POST['expected_keywords'] ?? ''); ?></textarea>
        </div>
    </div>

    <script>
        function toggleQuestionFields() {
            const type = document.getElementById('question_type').value;
            const mcqFields = document.getElementById('mcq_fields');
            const theoryFields = document.getElementById('theory_fields');

            mcqFields.style.display = 'none';
            theoryFields.style.display = 'none';

            if (type === 'MCQ') {
                mcqFields.style.display = 'block';
            } else if (type === 'Theory' || type === 'Fill-in-the-gaps') {
                theoryFields.style.display = 'block';
            }
        }

        // Call on page load to set initial state
        document.addEventListener('DOMContentLoaded', function () {
            toggleQuestionFields();
            updateQuestionTypes();
        });
    </script>
    <script>
        const examsMap = <?php echo json_encode($exams_map); ?>;

        function updateQuestionTypes() {
            const examSelect = document.getElementById('exam_id');
            const questionTypeSelect = document.getElementById('question_type');
            const selectedExamId = examSelect.value;
            const selectedExamMode = examsMap[selectedExamId];

            // Reset selected question type
            questionTypeSelect.value = '';
            toggleQuestionFields();

            // Show/hide question types based on exam mode
            document.querySelectorAll('.question-type-option').forEach(option => {
                const type = option.getAttribute('data-type');
                if (!selectedExamId || selectedExamMode === 'Mixed' || selectedExamMode === type) {
                    option.style.display = 'block';
                    option.disabled = false;
                } else {
                    option.style.display = 'none';
                    option.disabled = true;
                }
            });
        }
    </script>

    <button type="submit">Add Question</button>
    <!-- <input type="submit" value="Add Question"> -->
</form>

<h2 class="mt-20">Existing Questions</h2>
<?php $i = 1;
if (empty($questions)): ?>
    <p>No questions found in the bank.</p>
<?php else: ?>
    <table>
        <thead>
            <tr>
                <th>S/n</th>
                <th hidden>ID</th>
                <th>Exam</th>
                <th>Type</th>
                <th>Question Text</th>
                <th>Points</th>
                <th>Details/Answer</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($questions as $question): ?>
                <tr>
                    <td><?= $i++; ?></td>
                    <td hidden><?php echo htmlspecialchars($question['question_id']); ?></td>
                    <td><?php echo htmlspecialchars($question['exam_title']); ?></td>
                    <td><?php echo htmlspecialchars($question['question_type']); ?></td>
                    <td><?php echo htmlspecialchars(substr($question['question_text'], 0, 50)) . (strlen($question['question_text']) > 50 ? '...' : ''); ?>
                    </td>
                    <td><?php echo htmlspecialchars($question['points']); ?></td>
                    <td>
                        <?php if ($question['question_type'] === 'MCQ'): ?>
                            <strong>Correct:</strong> <?php echo htmlspecialchars($question['correct_option']); ?><br>
                            A: <?php echo htmlspecialchars($question['option_a']); ?><br>
                            B: <?php echo htmlspecialchars($question['option_b']); ?><br>
                            C: <?php echo htmlspecialchars($question['option_c']); ?><br>
                            D: <?php echo htmlspecialchars($question['option_d']); ?>
                        <?php elseif ($question['question_type'] === 'Fill-in-the-gaps'): ?>
                            <strong>Answer:</strong> <?php echo htmlspecialchars($question['correct_answer']); ?>
                        <?php elseif ($question['question_type'] === 'Theory'): ?>
                            <strong>Keywords:</strong> <?php echo htmlspecialchars($question['expected_keywords']); ?>
                        <?php else: ?>
                            N/A
                        <?php endif; ?>
                    </td>
                    <td>
                        <a
                            href="<?php echo BASE_URL; ?>admin/edit_question.php?id=<?php echo $question['question_id']; ?>">Edit</a>
                        |
                        <a href="<?php echo BASE_URL; ?>admin/questions.php?action=delete&id=<?php echo $question['question_id']; ?>"
                            onclick="return confirm('Are you sure you want to delete this question?');">Delete</a>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>
<?php
include 'includes/footer.php';
?>