<?php
// ai_utils.php - Utility functions for AI integration

// Ensure Composer autoloader is included (it is in db.php, but we include it here for standalone use if needed)
if (!defined('PROJECT_ROOT_DIR')) {
    require_once __DIR__ . '/db.php';
}

use OpenAI\OpenAI;

/**
 * Calls the OpenAI API to grade a theory question.
 * 
 * @param string $question_text The original question text.
 * @param string $correct_answer The model correct answer (or expected keywords/sentences).
 * @param string $student_answer The student's submitted answer.
 * @param float $max_points The maximum points available for the question.
 * @return array|null Returns an array with 'score' (float) and 'reason' (string), or null on failure.
 */
function ai_grade_theory_answer($question_text, $correct_answer, $student_answer, $max_points)
{
    // 1. Initialize OpenAI Client
    // The API key and base URL are pre-configured in the environment.
    try {
        $client = OpenAI::client(getenv('OPENAI_API_KEY'));
    } catch (\Exception $e) {
        error_log("OpenAI Client Initialization Failed: " . $e->getMessage());
        return null;
    }

    // 2. Construct the prompt
    $system_prompt = "You are an expert academic grader. Your task is to grade a student's answer to a theory question based on its correctness and similarity to the provided model answer. The grading must be fair and objective. You must return a JSON object with two keys: 'score' (a float between 0 and $max_points) and 'reason' (a brief explanation for the score).";

    $user_prompt = "Question: \"$question_text\"\n\n" .
        "Model Correct Answer/Key Concepts: \"$correct_answer\"\n\n" .
        "Student's Answer: \"$student_answer\"\n\n" .
        "Maximum Points: $max_points\n\n" .
        "Grade the student's answer and provide a score (float) and a reason (string) in a JSON object.";

    // 3. Call the API
    try {
        $response = $client->chat()->create([
            'model' => 'gpt-4.1-mini', // Using a capable and fast model
            'messages' => [
                ['role' => 'system', 'content' => $system_prompt],
                ['role' => 'user', 'content' => $user_prompt],
            ],
            'response_format' => ['type' => 'json_object'],
            'temperature' => 0.0, // Keep grading deterministic
        ]);

        // 4. Process the response
        $json_response = $response->choices[0]->message->content;
        $grading_result = json_decode($json_response, true);

        if (json_last_error() !== JSON_ERROR_NONE || !isset($grading_result['score']) || !isset($grading_result['reason'])) {
            error_log("AI Grading: Invalid JSON response from API: " . $json_response);
            return null;
        }

        // 5. Validate and sanitize score
        $score = (float) $grading_result['score'];
        $score = max(0.0, min($max_points, $score)); // Ensure score is within [0, max_points]

        return [
            'score' => $score,
            'reason' => $grading_result['reason']
        ];

    } catch (\Exception $e) {
        error_log("AI Grading API Call Failed: " . $e->getMessage());
        return null;
    }
}
?>