// proctoring.js - Handles webcam, MediaRecorder, micro-chunking, and Web Worker pool for parallel uploads.

// These constants will be defined in the take_exam.php file before this script is loaded.
let WORKER_SCRIPT_PATH;
let CHUNK_UPLOAD_HANDLER;
let EXAM_SUBMIT_HANDLER;
let FINALIZE_UPLOAD_HANDLER;
let MAX_WORKERS;
let CHUNK_DURATION_MS;
let TOTAL_QUESTIONS; // New constant

class ProctoringSystem {
    constructor(sessionId, videoElementId, submitButtonId) {
        this.sessionId = sessionId;
        this.videoElement = document.getElementById(videoElementId);
        this.submitButton = document.getElementById(submitButtonId);
        this.mediaRecorder = null;
        this.stream = null;
        this.chunkQueue = [];
        this.workerPool = [];
        this.nextChunkIndex = 0;
        this.isRecording = false;
        this.isSubmitting = false;
        this.uploadInProgress = 0;
        this.uploadFinished = false;

        this.initWorkerPool();
        this.submitButton.addEventListener('click', this.submitExam.bind(this));
    }

    initWorkerPool() {
        for (let i = 0; i < MAX_WORKERS; i++) {
            const worker = new Worker(WORKER_SCRIPT_PATH);
            worker.onmessage = this.handleWorkerMessage.bind(this);
            worker.onerror = this.handleWorkerError.bind(this);
            this.workerPool.push({ worker: worker, isBusy: false });
        }
        console.log(`Worker pool initialized with ${MAX_WORKERS} workers.`);
    }

    handleWorkerMessage(e) {
        const { status, success, chunkIndex, message } = e.data;

        // Find the worker that sent the message (based on the fact that only busy workers send messages)
        const workerEntry = this.workerPool.find(w => w.worker === e.target);

        if (status === 'complete' || status === 'error') {
            this.uploadInProgress--;
            if (workerEntry) {
                workerEntry.isBusy = false; // Mark worker as available
            }
            
            if (status === 'complete') {
                console.log(`Chunk ${chunkIndex} uploaded. Success: ${success}. Message: ${message}`);
            } else {
                console.error(`Chunk ${chunkIndex} upload failed. Message: ${message}`);
            }
            
            this.processQueue(); // Try to process the next chunk
        } else if (status === 'finalized') {
            this.uploadFinished = true;
            console.log(`Video upload finalization complete. Success: ${success}. Message: ${message}`);
        }
    }

    handleWorkerError(e) {
        console.error('Worker error:', e);
    }

    startProctoring() {
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            alert('Webcam not supported in this browser.');
            return;
        }

        navigator.mediaDevices.getUserMedia({ video: true, audio: false })
            .then(stream => {
                this.stream = stream;
                this.videoElement.srcObject = stream;
                this.videoElement.play();
                this.startRecording(stream);
            })
            .catch(err => {
                console.error('Error accessing webcam:', err);
                alert('Could not access webcam. Proctoring failed. Please ensure you have a webcam and have granted permission.');
            });
    }

    startRecording(stream) {
        const options = { mimeType: 'video/webm' }; 
        
        try {
            this.mediaRecorder = new MediaRecorder(stream, options);
        } catch (e) {
            console.error('MediaRecorder failed to start:', e);
            alert('MediaRecorder failed to initialize. Check browser support.');
            return;
        }

        this.mediaRecorder.ondataavailable = (event) => {
            if (event.data.size > 0) {
                this.chunkQueue.push({ chunk: event.data, index: this.nextChunkIndex++ });
                this.processQueue();
            }
        };

        this.mediaRecorder.onstop = () => {
            console.log('MediaRecorder stopped. Final chunks are in queue.');
            this.isRecording = false;
        };

        this.mediaRecorder.start(CHUNK_DURATION_MS); // Split into micro-chunks
        this.isRecording = true;
        console.log(`Recording started, chunking every ${CHUNK_DURATION_MS}ms.`);
    }

    processQueue() {
        if (this.chunkQueue.length === 0) {
            // If the queue is empty AND we are not recording AND we are submitting, 
            // it means all chunks have been generated. We can now check if all uploads are done.
            if (!this.isRecording && this.isSubmitting && this.uploadInProgress === 0 && !this.uploadFinished) {
                this.finalizeUpload();
            }
            return;
        }

        // Find an available worker
        const availableWorker = this.workerPool.find(w => !w.isBusy);

        if (availableWorker) {
            const chunkData = this.chunkQueue.shift();
            availableWorker.isBusy = true;
            this.uploadInProgress++;

            // Send the chunk to the worker
            availableWorker.worker.postMessage({
                chunk: chunkData.chunk,
                chunkIndex: chunkData.index,
                sessionId: this.sessionId,
                uploadUrl: CHUNK_UPLOAD_HANDLER,
                finalizeUrl: FINALIZE_UPLOAD_HANDLER
            });

            // The worker will send a message back when done, and we will mark it as available in handleWorkerMessage.
            // this.processQueue() is called in handleWorkerMessage.
            // We do NOT mark availableWorker.isBusy = false here. This is the fix.
        }
    }

    stopRecording() {
        if (this.mediaRecorder && this.isRecording) {
            this.mediaRecorder.stop();
            // Stop the stream tracks to release the camera
            this.stream.getTracks().forEach(track => track.stop());
            this.videoElement.srcObject = null;
        }
    }

    async submitExam() {
        if (this.isSubmitting) return;
        this.isSubmitting = true;
        this.submitButton.disabled = true;
        this.submitButton.textContent = 'Submitting...';

        // 1. Stop recorder immediately (as per requirement)
        this.stopRecording();

        // 2. Collect student answers
        const answers = this.collectAnswers();

        // 3. Prepare data for instant submission
        const formData = new FormData();
        formData.append('session_id', this.sessionId);
        formData.append('answers', JSON.stringify(answers));

        // 4. Perform instant submission (No waiting for video uploads)
        try {
            const response = await fetch(EXAM_SUBMIT_HANDLER, {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                console.log('Exam answers submitted instantly. Video upload continuing in background.');
                
                // 5. Instantly redirect to "Exam Submitted" page
                window.location.href = result.redirect_url;
                
            } else {
                alert('Error submitting exam answers: ' + result.message);
                this.isSubmitting = false;
                this.submitButton.disabled = false;
                this.submitButton.textContent = 'Submit Exam';
            }
        } catch (error) {
            console.error('Submission failed:', error);
            alert('An error occurred during submission. Please try again.');
            this.isSubmitting = false;
            this.submitButton.disabled = false;
            this.submitButton.textContent = 'Submit Exam';
        }
    }

    collectAnswers() {
        const answers = {};
        const form = document.getElementById('exam-form');
        if (!form) return answers;

        // Iterate over all question containers to collect answers
        const questionElements = form.querySelectorAll('.question-container');
        questionElements.forEach(qEl => {
            const questionId = qEl.dataset.questionId;
            if (!questionId) return;

            // Find the input element with the name attribute matching the question ID
            const inputName = `q_${questionId}`;
            let answer = null;

            // Handle radio buttons (MCQ)
            const radio = form.querySelector(`input[name="${inputName}"]:checked`);
            if (radio) {
                answer = radio.value;
            } else {
                // Handle text/textarea inputs (Theory/Fill-in-the-gaps)
                const textInput = form.querySelector(`[name="${inputName}"]`);
                if (textInput) {
                    answer = textInput.value;
                }
            }

            if (answer !== null) {
                answers[questionId] = answer;
            }
        });

        return answers;
    }

    finalizeUpload() {
        // Send a special message to one of the workers to handle the finalization call
        const worker = this.workerPool[0].worker;
        worker.postMessage({ chunk: 'FINALIZE', sessionId: this.sessionId, finalizeUrl: FINALIZE_UPLOAD_HANDLER });
    }
}

// Global function to initialize the system
window.initProctoring = function(sessionId, paths) {
    // Set constants from the paths object passed from PHP
    WORKER_SCRIPT_PATH = paths.WORKER_SCRIPT_PATH;
    CHUNK_UPLOAD_HANDLER = paths.CHUNK_UPLOAD_HANDLER;
    EXAM_SUBMIT_HANDLER = paths.EXAM_SUBMIT_HANDLER;
    FINALIZE_UPLOAD_HANDLER = paths.FINALIZE_UPLOAD_HANDLER;
    MAX_WORKERS = paths.MAX_WORKERS;
    CHUNK_DURATION_MS = paths.CHUNK_DURATION_MS;
    TOTAL_QUESTIONS = paths.TOTAL_QUESTIONS;
    
    // Check if we are on the take_exam page before initializing
    if (document.getElementById('webcam-feed') && document.getElementById('submit-exam-btn')) {
        const proctoringSystem = new ProctoringSystem(sessionId, 'webcam-feed', 'submit-exam-btn');
        proctoringSystem.startProctoring();
        
        // Ensure all chunks are uploaded before the browser closes
        window.addEventListener('beforeunload', (event) => {
            if (proctoringSystem.isRecording || proctoringSystem.chunkQueue.length > 0 || proctoringSystem.uploadInProgress > 0) {
                event.returnValue = 'Your exam is still in progress or video chunks are still uploading. Are you sure you want to leave?';
            }
        });
    }
};
