<?php
// submit_exam.php - Handles the instant submission of student answers.

header('Content-Type: application/json');

require_once '../db.php';
require_student();
require_once '../proctor/video_processor.php'; // Include video processing functions

$response = ['success' => false, 'message' => 'An unknown error occurred.', 'redirect_url' => BASE_URL . 'student/start_exam.php'];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = 'Invalid request method.';
    echo json_encode($response);
    exit;
}

// 1. Get and validate input
$session_id = (int)($_POST['session_id'] ?? 0);
$answers_json = $_POST['answers'] ?? null;
$student_id = get_current_user_id();

if ($session_id === 0 || $answers_json === null) {
    $response['message'] = 'Missing session ID or answers data.';
    echo json_encode($response);
    exit;
}

$answers = json_decode($answers_json, true);

if ($answers === null) {
    $response['message'] = 'Invalid answers data format.';
    echo json_encode($response);
    exit;
}

// 2. Fetch session and exam details
$sql_session = "SELECT S.exam_id, E.exam_mode, E.duration_minutes
                FROM student_exam_sessions S
                JOIN exams E ON S.exam_id = E.exam_id
                WHERE S.session_id = ? AND S.student_id = ? AND S.is_submitted = FALSE";
$result_session = execute_query($mysqli, $sql_session, 'ii', [$session_id, $student_id]);

if (!$result_session || $result_session->num_rows === 0) {
    $response['message'] = 'Invalid, unauthorized, or already submitted exam session.';
    echo json_encode($response);
    exit;
}

$session_data = $result_session->fetch_assoc();
$exam_id = $session_data['exam_id'];
$exam_mode = $session_data['exam_mode'];
$total_score = 0.00;

// 3. Start Transaction
$mysqli->begin_transaction();

try {
    // 4. Fetch all questions for the exam
    $sql_questions = "SELECT question_id, question_type, correct_option, correct_answer, points
                      FROM exam_questions
                      WHERE exam_id = ?";
    $result_questions = execute_query($mysqli, $sql_questions, 'i', [$exam_id]);
    $questions = [];
    if ($result_questions) {
        while ($row = $result_questions->fetch_assoc()) {
            $questions[$row['question_id']] = $row;
        }
    }

    // 5. Process and save answers
    $sql_insert_answer = "INSERT INTO student_answers (session_id, question_id, student_answer, is_correct, score_earned)
                          VALUES (?, ?, ?, ?, ?)
                          ON DUPLICATE KEY UPDATE student_answer = VALUES(student_answer), is_correct = VALUES(is_correct), score_earned = VALUES(score_earned)";
    
    foreach ($answers as $question_id => $student_answer) {
        $question_id = (int)$question_id;
        $student_answer = trim($student_answer);
        
        if (!isset($questions[$question_id])) {
            // Skip answers for questions not found in the exam
            continue;
        }

        $question = $questions[$question_id];
        $is_correct = null;
        $score_earned = 0.00;
        $points = (float)$question['points'];

        // Grading logic (only for auto-gradable types)
        if ($question['question_type'] === 'MCQ') {
            $is_correct = (strtoupper($student_answer) === strtoupper($question['correct_option']));
            $score_earned = $is_correct ? $points : 0.00;
        } elseif ($question['question_type'] === 'Fill-in-the-gaps') {
            // Fill-in-the-gaps: Case-insensitive exact match
            // Use strtolower for case-insensitive comparison as requested
            $is_correct = (strtolower($student_answer) === strtolower($question['correct_answer']));
            $score_earned = $is_correct ? $points : 0.00;
        } elseif ($question['question_type'] === 'Theory') {
            // Theory: Not auto-graded. Set score to 0 and is_correct to 0 (false) for now.
            // Manual grading will update score_earned and is_correct later.
            $is_correct = 0; // Set to 0 (false) as it's not auto-graded as correct
            $score_earned = 0.00;
        } else {
            // Default for non-auto-graded types (should not happen if logic is correct)
            $is_correct = 0;
            $score_earned = 0.00;
        }

	        // Insert/Update answer
	        // The is_correct field (BOOLEAN/TINYINT) should be bound as an integer (i).
	        // The score_earned field (DECIMAL/FLOAT) should be bound as a double (d).
	        // Since $is_correct can be NULL for Theory questions, we must handle it.
	        // We will use 'i' for is_correct and rely on PHP's casting of NULL to 0 for integer binding,
	        // which is acceptable for a boolean/tinyint column if we assume 0 means false/null.
	        // The correct type string for 5 parameters (i, i, s, i, d) is 'iisi d'.
	        // Let's use 'iisi d' and ensure $is_correct is cast to int.
	        
	        // Correct type string for 5 parameters (i, i, s, i, d) is 'iisid'.
	        // Since $is_correct is now always 0 or 1, we can safely bind it as an integer.
	        $insert_result = execute_non_query($mysqli, $sql_insert_answer, 'iisid', 
	            [$session_id, $question_id, $student_answer, (int)$is_correct, $score_earned]
	        );
        if ($insert_result === false) {
            throw new Exception("Failed to save answer for question ID: $question_id");
        }

        $total_score += $score_earned;
    }

    // 6. Update session as submitted
    // CRITICAL FIX: Ensure end_time is set to the exact moment of submission
    $sql_update_session = "UPDATE student_exam_sessions
                           SET is_submitted = TRUE, end_time = NOW(), total_score = ?
                           WHERE session_id = ? AND student_id = ?";
    $update_result = execute_non_query($mysqli, $sql_update_session, 'dii', [$total_score, $session_id, $student_id]);

    if ($update_result === false) {
        throw new Exception("Failed to update exam session status.");
    }
    // 7. Commit transaction
    $mysqli->commit();

    // 8. Trigger video processing in the background
    // This ensures the video is processed immediately upon submission/timeout.
    process_video_background($session_id);

    // 9. Success response
    $response['success'] = true;
    $response['message'] = 'Exam submitted successfully. Video upload is continuing in the background.';
    $response['redirect_url'] = BASE_URL . 'student/exam_submitted.php?session_id=' . $session_id;

} catch (Exception $e) {
    // 9. Rollback and error response
    $mysqli->rollback();
	    $response['message'] = 'Submission failed: ' . $e->getMessage() . '. SQL Error: ' . $mysqli->error;
    error_log($response['message']);
}

$mysqli->close();
echo json_encode($response);
?>
