<?php
// take_exam.php - Student Exam Interface with Proctoring (Single Question View)

require_once '../db.php';
require_student();

$session_id = (int) ($_GET['session_id'] ?? 0);
$student_id = get_current_user_id();

if ($session_id === 0) {
    redirect('student/start_exam.php');
}

// 1. Fetch Session and Exam Details
$sql_session = "SELECT S.*, E.exam_title, E.duration_minutes, E.exam_mode, E.total_questions, E.shuffle_questions, S.public_session_id, S.question_order
                FROM student_exam_sessions S
                JOIN exams E ON S.exam_id = E.exam_id
                WHERE S.session_id = ? AND S.student_id = ?";
$result_session = execute_query($mysqli, $sql_session, 'ii', [$session_id, $student_id]);

if (!$result_session || $result_session->num_rows === 0) {
    $_SESSION['error_message'] = "Invalid or unauthorized exam session.";
    redirect('student/start_exam.php');
}

$session_data = $result_session->fetch_assoc();

if ($session_data['is_submitted']) {
    $_SESSION['error_message'] = "This exam has already been submitted. Your score is: " . $session_data['total_score'] . "%";
    redirect('student/exam_submitted.php?session_id=' . $session_id);
}

// Calculate remaining time for the countdown timer (Server-side time validation)
$start_time = strtotime($session_data['start_time']);
$duration_seconds = $session_data['duration_minutes'] * 60;
$time_elapsed = time() - $start_time;
$time_remaining = $duration_seconds - $time_elapsed;

if ($time_remaining <= 0) {
    // CRITICAL FIX: Auto-submit if time is up. Set end_time before redirecting.
    $sql_update_session = "UPDATE student_exam_sessions
                           SET is_submitted = TRUE, end_time = NOW()
                           WHERE session_id = ? AND student_id = ?";
    execute_non_query($mysqli, $sql_update_session, 'ii', [$session_id, $student_id]);
    
    // NOTE: The score calculation is handled by the client-side JS calling submit_exam.php
    // However, if the page is just refreshed after timeout, the JS won't run.
    // The safest way is to redirect to a page that handles the final score calculation.
    
    $_SESSION['error_message'] = "Time is up! The exam has been automatically submitted.";
    redirect('student/exam_submitted.php?session_id=' . $session_id);
}

// 2. Fetch and Order Questions with Session-based Shuffle
$exam_id = $session_data['exam_id'];
$limit_questions = (int)$session_data['total_questions'];
$shuffle_enabled = (int)($session_data['shuffle_questions'] ?? 0);
$stored_question_order = $session_data['question_order'];
$questions = []; // Initialize questions array

// Check if question order is already stored in the session
if (!empty($stored_question_order)) {
    // Use stored order (for consistency across page refreshes)
    $question_ids = explode(',', $stored_question_order);
    
    // Defensive check: Truncate question IDs to the exam's limit in case of corrupted data
    if (count($question_ids) > $limit_questions) {
        $question_ids = array_slice($question_ids, 0, $limit_questions);
    }
    
    $question_ids_str = implode(',', array_map('intval', $question_ids)); // Sanitize
    
    // Fetch questions in the stored order using FIELD() function
    $sql_questions = "SELECT Q.*, A.student_answer
                      FROM exam_questions Q
                      LEFT JOIN student_answers A ON Q.question_id = A.question_id AND A.session_id = ?
                      WHERE Q.exam_id = ? AND Q.question_id IN ($question_ids_str)
                      ORDER BY FIELD(Q.question_id, $question_ids_str)";
    $result_questions = execute_query($mysqli, $sql_questions, 'ii', [$session_id, $exam_id]);
    $questions = $result_questions ? $result_questions->fetch_all(MYSQLI_ASSOC) : [];
} else {
    // First time loading - determine order and store it
    $order_clause = $shuffle_enabled ? "ORDER BY RAND()" : "ORDER BY Q.question_id";
    
    $sql_questions = "SELECT Q.*, A.student_answer
                      FROM exam_questions Q
                      LEFT JOIN student_answers A ON Q.question_id = A.question_id AND A.session_id = ?
                      WHERE Q.exam_id = ?
                      {$order_clause}
                      LIMIT {$limit_questions}";
    $result_questions = execute_query($mysqli, $sql_questions, 'ii', [$session_id, $exam_id]);
    $questions = $result_questions ? $result_questions->fetch_all(MYSQLI_ASSOC) : [];
    
    // Store the question order in the session
    $question_ids = array_column($questions, 'question_id');
    $question_order_str = implode(',', $question_ids);
    
    // Update the session with the question order
    // Check if column exists first
    $sql_check_column = "SHOW COLUMNS FROM student_exam_sessions LIKE 'question_order'";
    $column_result = execute_query($mysqli, $sql_check_column);
    
    if ($column_result && $column_result->num_rows === 0) {
        // Add the column if it doesn't exist
        $sql_add_column = "ALTER TABLE student_exam_sessions ADD COLUMN question_order TEXT DEFAULT NULL AFTER video_upload_complete";
        $mysqli->query($sql_add_column);
    }
    
    $sql_update_order = "UPDATE student_exam_sessions SET question_order = ? WHERE session_id = ?";
    execute_non_query($mysqli, $sql_update_order, 'si', [$question_order_str, $session_id]);
}

// The total number of questions is the count of the questions array, which is correctly limited on first load
// and correctly loaded from the stored order on subsequent loads.
// The final count should be the number of questions loaded, which is already limited by $limit_questions
// on the first load, and by the stored question_order on subsequent loads.
$total_questions = count($questions);

// 3. Get current question index from session or default to 0
$current_question_index = isset($_SESSION['exam_' . $session_id . '_current_q']) ? (int)$_SESSION['exam_' . $session_id . '_current_q'] : 0;

// Validate the index
if ($current_question_index < 0 || $current_question_index >= $total_questions) {
    // CRITICAL FIX: If the index is invalid, reset it AND clear the session variable
    // to prevent the total question count from being based on a bad index.
    unset($_SESSION['exam_' . $session_id . '_current_q']);
    $current_question_index = 0;
}

// Define JS constants
$js_paths = [
    'WORKER_SCRIPT_PATH' => WORKER_SCRIPT_PATH,
    'CHUNK_UPLOAD_HANDLER' => CHUNK_UPLOAD_HANDLER,
    'EXAM_SUBMIT_HANDLER' => EXAM_SUBMIT_HANDLER,
    'FINALIZE_UPLOAD_HANDLER' => BASE_URL . 'proctor/finalize_upload.php',
    'MAX_WORKERS' => MAX_WORKERS,
    'CHUNK_DURATION_MS' => CHUNK_DURATION_MS,
    'TOTAL_QUESTIONS' => $total_questions,
    'SESSION_ID' => $session_id,
    'CURRENT_QUESTION_INDEX' => $current_question_index
];

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($session_data['exam_title']); ?> - Exam</title>
    <link rel="stylesheet" href="<?php echo MAIN_CSS_PATH; ?>">
    <style>
        /* Basic styles for single question view */
        .question-container {
            display: none;
        }

        .question-container.active {
            display: block;
        }
    </style>
</head>

<body>
    <div class="exam-interface">

        <!-- Sidebar for Timer and Proctoring Feed -->
        <div class="exam-sidebar">
            <h2>Time Remaining</h2>
            <div id="timer" data-time-remaining="<?php echo $time_remaining; ?>"></div>

            <h2>Proctoring Feed</h2>
            <video id="webcam-feed" autoplay muted></video>
            <p class="proctoring-info"><small>Your webcam is being recorded for proctoring purposes.</small></p>
        </div>

        <!-- Main Exam Content -->
        <div class="exam-main">

            <div class="exam-header">
                <h2><?php echo htmlspecialchars($session_data['exam_title']); ?></h2>
                <p><strong>Session ID:</strong> <?php echo htmlspecialchars($session_data['public_session_id']); ?></p>
                <p class="question-counter">Question <span id="current-q-num"><?php echo $current_question_index + 1; ?></span> of
                    <?php echo $total_questions; ?>
                </p>
            </div>

            <form id="exam-form">
                <?php $q_num = 1;
                foreach ($questions as $question): ?>
                    <div class="question-container <?php echo ($q_num - 1 === $current_question_index) ? 'active' : ''; ?>" 
                         data-question-id="<?php echo $question['question_id']; ?>"
                         data-q-index="<?php echo $q_num - 1; ?>">
                        <h3>Question <?php echo $q_num++; ?> (<?php echo $question['question_type']; ?>)</h3>
                        <p><?php echo nl2br(htmlspecialchars($question['question_text'])); ?></p>

                        <?php if ($question['question_type'] === 'MCQ'): ?>
                            <?php
                            $options = ['A' => $question['option_a'], 'B' => $question['option_b'], 'C' => $question['option_c'], 'D' => $question['option_d']];
                            // CRITICAL FIX: Ensure student_answer is treated as a string for comparison
                            $student_answer = (string)($question['student_answer'] ?? '');
                            ?>
                            <?php foreach ($options as $key => $value):
                                if (!empty($value)): ?>
                                    <label>
                                        <input type="radio" name="q_<?php echo $question['question_id']; ?>" value="<?php echo $key; ?>"
                                            <?php echo ($student_answer === $key) ? 'checked' : ''; ?>>
                                        <?php echo htmlspecialchars($key); ?>. <?php echo htmlspecialchars($value); ?>
                                    </label><br>
                                <?php endif; endforeach; ?>
                        <?php elseif ($question['question_type'] === 'Theory'): ?>
                            <textarea name="q_<?php echo $question['question_id']; ?>"
                                rows="5"><?php echo htmlspecialchars((string)($question['student_answer'] ?? '')); ?></textarea>
                        <?php elseif ($question['question_type'] === 'Fill-in-the-gaps'): ?>
                            <input type="text" name="q_<?php echo $question['question_id']; ?>"
                                value="<?php echo htmlspecialchars((string)($question['student_answer'] ?? '')); ?>">
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>

                <div class="navigation-buttons">
                    <button type="button" id="prev-btn" <?php echo ($current_question_index === 0) ? 'disabled' : ''; ?>>Previous</button>
                    <button type="button" id="skip-btn" <?php echo ($current_question_index === $total_questions - 1) ? 'style="display:none;"' : ''; ?>>Skip</button>
                    <button type="button" id="next-btn" <?php echo ($current_question_index === $total_questions - 1) ? 'style="display:none;"' : ''; ?>>Next</button>
                    <button type="button" id="submit-exam-btn" <?php echo ($current_question_index === $total_questions - 1) ? 'style="display:inline-block;"' : 'style="display:none;"'; ?>>Submit Exam Instantly</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Pass PHP constants to JavaScript
        const jsPaths = <?php echo json_encode($js_paths); ?>;
    </script>
    <script src="<?php echo MAIN_JS_PATH; ?>"></script>
    <script>
        // Initialize the proctoring system and start the timer
        document.addEventListener('DOMContentLoaded', function () {
            // 1. Initialize Proctoring
            window.initProctoring(<?php echo $session_id; ?>, jsPaths);

            // 2. Start Countdown Timer
            const timerEl = document.getElementById('timer');
            let timeRemaining = parseInt(timerEl.dataset.timeRemaining);

            function updateTimer() {
                const minutes = Math.floor(timeRemaining / 60);
                const seconds = timeRemaining % 60;
                timerEl.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;

                if (timeRemaining <= 0) {
                    clearInterval(timerInterval);
                    alert('Time is up! The exam will be submitted automatically.');
                    // Auto-submit the exam
                    document.getElementById('submit-exam-btn').click();
                } else {
                    timeRemaining--;
                }
            }

            updateTimer();
            const timerInterval = setInterval(updateTimer, 1000);

            // 3. Single Question View Logic with Auto-Save
            const questionContainers = document.querySelectorAll('.question-container');
            const totalQuestions = questionContainers.length;
            let currentQuestionIndex = jsPaths.CURRENT_QUESTION_INDEX; // Load from server

            const prevBtn = document.getElementById('prev-btn');
            const nextBtn = document.getElementById('next-btn');
            const skipBtn = document.getElementById('skip-btn');
            const submitBtn = document.getElementById('submit-exam-btn');
            const currentQNumEl = document.getElementById('current-q-num');

            // Auto-save function
            function autoSaveCurrentAnswer() {
                const currentContainer = questionContainers[currentQuestionIndex];
                const questionId = currentContainer.dataset.questionId;
                let answer = '';

                // Get the answer based on question type
                const radioInput = currentContainer.querySelector('input[type="radio"]:checked');
                const textInput = currentContainer.querySelector('input[type="text"]');
                const textareaInput = currentContainer.querySelector('textarea');

                if (radioInput) {
                    answer = radioInput.value;
                } else if (textInput) {
                    answer = textInput.value;
                } else if (textareaInput) {
                    answer = textareaInput.value;
                }

                // Send AJAX request to save answer
                fetch('<?php echo BASE_URL; ?>student/save_answer.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `session_id=${jsPaths.SESSION_ID}&question_id=${questionId}&answer=${encodeURIComponent(answer)}&current_index=${currentQuestionIndex}`
                }).catch(err => console.error('Auto-save error:', err));
            }

            function showQuestion(index) {
                // Auto-save before changing question
                autoSaveCurrentAnswer();

                questionContainers.forEach((container, i) => {
                    container.classList.remove('active');
                });
                questionContainers[index].classList.add('active');
                currentQuestionIndex = index;
                currentQNumEl.textContent = index + 1;

                // Update button states
                prevBtn.disabled = index === 0;

                if (index === totalQuestions - 1) {
                    nextBtn.style.display = 'none';
                    skipBtn.style.display = 'none';
                    submitBtn.style.display = 'inline-block';
                } else {
                    nextBtn.style.display = 'inline-block';
                    skipBtn.style.display = 'inline-block';
                    submitBtn.style.display = 'none';
                }
            }

            prevBtn.addEventListener('click', () => {
                if (currentQuestionIndex > 0) {
                    showQuestion(currentQuestionIndex - 1);
                }
            });

            nextBtn.addEventListener('click', () => {
                if (currentQuestionIndex < totalQuestions - 1) {
                    showQuestion(currentQuestionIndex + 1);
                }
            });

            skipBtn.addEventListener('click', () => {
                if (currentQuestionIndex < totalQuestions - 1) {
                    showQuestion(currentQuestionIndex + 1);
                }
            });

            // Auto-save on input change
            questionContainers.forEach(container => {
                const inputs = container.querySelectorAll('input, textarea');
                inputs.forEach(input => {
                    input.addEventListener('change', autoSaveCurrentAnswer);
                    if (input.tagName === 'TEXTAREA' || input.type === 'text') {
                        input.addEventListener('input', debounce(autoSaveCurrentAnswer, 1000));
                    }
                });
            });

            // Debounce function for text inputs
            function debounce(func, wait) {
                let timeout;
                return function executedFunction(...args) {
                    const later = () => {
                        clearTimeout(timeout);
                        func(...args);
                    };
                    clearTimeout(timeout);
                    timeout = setTimeout(later, wait);
                };
            }

            // Initial display (already set by PHP, but ensure JS is in sync)
            if (totalQuestions > 0) {
                // Question is already shown by PHP, just ensure JS state is correct
                currentQNumEl.textContent = currentQuestionIndex + 1;
            }
        });
    </script>
</body>

</html>
<?php $mysqli->close(); ?>
